<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\News;
use Golonka\BBCode\BBCodeParser;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class NewsController extends Controller
{
    protected $bbcode;

    public function __construct(BBCodeParser $bbcode)
    {
        $this->bbcode = $bbcode;
    }

    public function index()
    {
        $news = News::with('author')->latest()->paginate(10);
        return view('admin.news.index', compact('news'));
    }

    public function create()
    {
        return view('admin.news.create');
    }

    public function store(Request $request)
    {
        Log::info('News article store method started');
        
        $validated = $request->validate([
            'title' => 'required|max:255',
            'short_description' => 'required|max:500',
            'content' => 'required',
            'image' => 'image|mimes:jpeg,png,jpg,gif|max:2048',
        ]);

        Log::info('Validation passed');
        
        $validated['author_id'] = Auth::id();
        Log::info('Author ID set: ' . Auth::id());

        $slug = Str::slug($validated['title']) ?: uniqid('news_');
        $validated['slug'] = $slug;
        Log::info('Slug generated: ' . $slug);

        // Parse BBCode to HTML
        try {
            $validated['content'] = $this->bbcode->parse($validated['content']);
            Log::info('BBCode parsed successfully');
        } catch (\Exception $e) {
            Log::error('BBCode parsing error: ' . $e->getMessage());
        }

        // Add storage debug info
        Log::info('Storage debug info', [
            'storage_path' => storage_path('app/public'),
            'storage_exists' => is_dir(storage_path('app/public')),
            'storage_writable' => is_writable(storage_path('app/public')),
            'news_dir_exists' => is_dir(storage_path('app/public/news')),
            'news_dir_writable' => is_writable(storage_path('app/public/news')),
            'php_user' => get_current_user(),
            'process_user' => function_exists('posix_getpwuid') && function_exists('posix_geteuid') 
                ? (posix_getpwuid(posix_geteuid())['name'] ?? 'unknown') 
                : 'unavailable'
        ]);

        // Handle image upload
        if ($request->hasFile('image')) {
            Log::info('Image found in request');
            
            $image = $request->file('image');
            
            // Log file details
            Log::info('Image details', [
                'original_name' => $image->getClientOriginalName(),
                'mime_type' => $image->getClientMimeType(),
                'size' => $image->getSize(),
                'extension' => $image->getClientOriginalExtension(),
                'is_valid' => $image->isValid()
            ]);
            
            if ($image->isValid()) {
                Log::info('Image is valid');
                
                // Generate filename
                $filename = time() . '_' . $slug . '.' . $image->getClientOriginalExtension();
                Log::info('Generated filename: ' . $filename);
                
                // Create directory if not exists
                $newsDir = 'news';
                if (!Storage::disk('public')->exists($newsDir)) {
                    Log::info('Creating news directory');
                    Storage::disk('public')->makeDirectory($newsDir);
                }
                
                try {
                    // Method 1: Using storeAs
                    $path = $image->storeAs($newsDir, $filename, 'public');
                    Log::info('storeAs method - Path returned: ' . var_export($path, true));
                    
                    if ($path) {
                        // Check if file actually exists
                        $fileExists = Storage::disk('public')->exists($path);
                        Log::info('File exists after storeAs: ' . ($fileExists ? 'yes' : 'no'));
                        
                        if ($fileExists) {
                            $validated['image'] = $path;
                            Log::info('Image successfully stored using storeAs: ' . $path);
                        } else {
                            Log::error('storeAs returned path but file does not exist');
                            $validated['image'] = null;
                        }
                    } else {
                        Log::info('storeAs failed, trying alternate method');
                        
                        // Method 2: Direct file operations
                        $fullPath = storage_path('app/public/' . $newsDir . '/' . $filename);
                        $directory = dirname($fullPath);
                        
                        Log::info('Trying direct file operations', [
                            'full_path' => $fullPath,
                            'directory' => $directory,
                            'directory_exists' => is_dir($directory),
                            'directory_writable' => is_dir($directory) ? is_writable($directory) : false
                        ]);
                        
                        // Ensure directory exists
                        if (!is_dir($directory)) {
                            $created = mkdir($directory, 0755, true);
                            Log::info('Directory creation result: ' . ($created ? 'success' : 'failed'));
                        }
                        
                        // Move the file
                        if ($image->move($directory, $filename)) {
                            $relativePath = $newsDir . '/' . $filename;
                            
                            // Verify file exists
                            if (Storage::disk('public')->exists($relativePath)) {
                                $validated['image'] = $relativePath;
                                Log::info('Image successfully stored using move: ' . $relativePath);
                            } else {
                                Log::error('File moved but not found in storage');
                                $validated['image'] = null;
                            }
                        } else {
                            Log::error('Failed to move uploaded file');
                            $validated['image'] = null;
                        }
                    }
                } catch (\Exception $e) {
                    Log::error('Exception during file storage: ' . $e->getMessage(), [
                        'exception_class' => get_class($e),
                        'file' => $e->getFile(),
                        'line' => $e->getLine(),
                        'trace' => $e->getTraceAsString()
                    ]);
                    $validated['image'] = null;
                }
            } else {
                Log::error('Uploaded image is not valid');
                $validated['image'] = null;
            }
        } else {
            Log::info('No image found in request');
            $validated['image'] = null;
        }
        
        // Debug final data before creation
        Log::info('Final validated data before database insert', $validated);
        
        try {
            $news = News::create($validated);
            Log::info('News article created with ID: ' . $news->id);
            
            // Verify what was actually saved
            $savedNews = News::find($news->id);
            Log::info('Saved news image value: ' . ($savedNews->image ?? 'null'));
        } catch (\Exception $e) {
            Log::error('Database error during news creation: ' . $e->getMessage());
            return redirect()->back()->withInput()->withErrors(['database' => 'Error saving article: ' . $e->getMessage()]);
        }

        return redirect()->route('admin.news.index')
            ->with('success', 'News article created successfully.');
    }

    public function show(News $news)
    {
        return view('admin.news.show', compact('news'));
    }

    public function edit(News $news)
    {
        return view('admin.news.edit', compact('news'));
    }

    public function update(Request $request, News $news)
    {
        Log::info('News article update method started for ID: ' . $news->id);
        
        $validated = $request->validate([
            'title' => 'required|max:255',
            'short_description' => 'required|max:500',
            'content' => 'required',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
        ]);

        Log::info('Update validation passed');

        $slug = Str::slug($validated['title']) ?: uniqid('news_');
        $validated['slug'] = $slug;
        Log::info('Update slug generated: ' . $slug);

        // Parse BBCode to HTML
        try {
            $validated['content'] = $this->bbcode->parse($validated['content']);
            Log::info('Update BBCode parsed successfully');
        } catch (\Exception $e) {
            Log::error('Update BBCode parsing error: ' . $e->getMessage());
        }

        // Handle image upload for update
        if ($request->hasFile('image')) {
            Log::info('New image found in update request');
            
            // Delete old image if it exists
            if ($news->image && Storage::disk('public')->exists($news->image)) {
                Log::info('Deleting old image: ' . $news->image);
                Storage::disk('public')->delete($news->image);
            }

            $image = $request->file('image');
            
            if ($image->isValid()) {
                $filename = time() . '_' . $slug . '.' . $image->getClientOriginalExtension();
                Log::info('Update - Generated filename: ' . $filename);
                
                $newsDir = 'news';
                if (!Storage::disk('public')->exists($newsDir)) {
                    Storage::disk('public')->makeDirectory($newsDir);
                }
                
                try {
                    $path = $image->storeAs($newsDir, $filename, 'public');
                    
                    if ($path && Storage::disk('public')->exists($path)) {
                        $validated['image'] = $path;
                        Log::info('Update - Image successfully stored: ' . $path);
                    } else {
                        // Fallback method for update
                        $fullPath = storage_path('app/public/' . $newsDir . '/' . $filename);
                        $directory = dirname($fullPath);
                        
                        if (!is_dir($directory)) {
                            mkdir($directory, 0755, true);
                        }
                        
                        if ($image->move($directory, $filename)) {
                            $relativePath = $newsDir . '/' . $filename;
                            if (Storage::disk('public')->exists($relativePath)) {
                                $validated['image'] = $relativePath;
                                Log::info('Update - Image stored using fallback method: ' . $relativePath);
                            }
                        }
                    }
                } catch (\Exception $e) {
                    Log::error('Update - Exception during file storage: ' . $e->getMessage());
                }
            }
        }

        try {
            $news->update($validated);
            Log::info('News article updated successfully');
        } catch (\Exception $e) {
            Log::error('Database error during news update: ' . $e->getMessage());
            return redirect()->back()->withInput()->withErrors(['database' => 'Error updating article: ' . $e->getMessage()]);
        }

        return redirect()->route('admin.news.index')
            ->with('success', 'News article updated successfully.');
    }

    public function destroy(News $news)
    {
        Log::info('Deleting news article with ID: ' . $news->id);
        
        // Delete associated image
        if ($news->image && Storage::disk('public')->exists($news->image)) {
            Log::info('Deleting image: ' . $news->image);
            Storage::disk('public')->delete($news->image);
        }

        try {
            $news->delete();
            Log::info('News article deleted successfully');
        } catch (\Exception $e) {
            Log::error('Error deleting news article: ' . $e->getMessage());
            return redirect()->back()->withErrors(['delete' => 'Error deleting article: ' . $e->getMessage()]);
        }

        return redirect()->route('admin.news.index')
            ->with('success', 'News article deleted successfully.');
    }
}